/*
 * 
 * Created on Apr 17, 2005
 * by ccfisch
 */

import java.util.*;
import monte.*;
import cern.colt.corejava.*;
import java.net.*;

/**
 * This is a java implementation of the original fortran code MC.f used in MIT's 3.320
 * "Atomisitic Simulation of Materials" . The original code was written by Gerd Ceder
 * and updated by 3.320 TA's over the years.  
 * 
 * @see MCSampler for details about how the Monte Carlo trajectory (Importance Sampling)
 * is implemented. 
 * 
 * Runs MC over loops in chemical potential or temperature printing output
 * as specified by the input file provided on the command line. 
 * 
 * @author ccfisch
 * Date: Apr 24, 2005
 *
 */
public class RunMC {
    
    /**
     * @param args
     */
    public static void main(String[] args) {
        char nl = DataPrinter.nl;
        if(args==null){
            System.out.print("usage: RunMC <inputfile>"+nl+"       RunMC -h");
        }
        if(args[0].equals("-h")){
            System.out.print("usage: RunMC <inputfile>"+nl+"       RunMC -h"+nl);   
            System.out.println("RunMC: run grand-canonical Monte Carlo on a 2-D square lattice"+
                            nl+"       of spin occupation variables (+/- 1) and a Hamiltonian"+
                            nl+"       with nearest-neighbor and 2nd-nn interactions"+nl);
            InputReader in = new InputReader();
            return;
        }
        try{
            System.out.println("-------------Job/System Properties--------------------");
            System.out.println("  started on: "+(new Date()).toString());
            System.out.println("    hostname: "+InetAddress.getLocalHost().getHostName());
            System.out.println("executing on: "+System.getProperty("os.name").toString());
            System.out.println("        arch: "+System.getProperty("os.arch"));
            System.out.println("      kernel: "+System.getProperty("os.version"));
            System.out.println(" JVM-version: "+System.getProperty("java.vm.version"));
            System.out.println("  JVM-vender: "+System.getProperty("java.vm.vender"));
            System.out.println("    JVM-name: "+System.getProperty("java.vm.name"));
            System.out.println("-----------------------------------------------------");
            InputReader input = new InputReader(args[0]);
            long beg = System.currentTimeMillis();
            runLoops(input);
            long end = System.currentTimeMillis();
            System.out.println("#MC run took "+(double)(end-beg)/1000+" sec");
        }catch (Exception e){
            System.out.println(e.getMessage());
            e.printStackTrace();
        }
    }
    
    /**
     * performs loops over mu and temp according to parameters set in InputReader
     * @param in
     */
    public static void runLoops(InputReader in){
        
        final int SIZE=in.m_SIZE;

        System.out.println("#System size: "+SIZE);
        System.out.println("#Equilibration Passes: "+in.m_NUM_EQ);
        System.out.println("#Number of Passes: "+in.m_NUM_PASS);
        System.out.println("#NN interaction: "+in.m_V1);
        System.out.println("#2NN interaction: "+in.m_V2);
        
        final boolean tempIS_inside_loop=in.m_T_INSIDE;        
        int[][] config = getConfig(SIZE,in.m_INIT_CONFIG);
        
        double T_lo=in.m_T_FINAL;
        double T_hi=in.m_T_START;
        if(in.m_T_START<in.m_T_FINAL){
            T_lo=in.m_T_START;
            T_hi=in.m_T_FINAL;
        }
        double MU_lo=in.m_MU_FINAL;
        double MU_hi=in.m_MU_START;
        if(in.m_MU_START<in.m_MU_FINAL){
            MU_lo=in.m_MU_START;
            MU_hi=in.m_MU_FINAL;
        }
        
        double temp=in.m_T_START;
        double mu=in.m_MU_START;
        //double en = getEnergy(config,V1,V2,mu);
        double en = MCSampler.getEnergy(config,in.m_V1,in.m_V2,mu);
        System.out.println("#initial configuration: "+in.m_INIT_CONFIG);
        System.out.println("#energy of initial config: "+en);
        System.out.println("#energy/site of initial config: "+en/(SIZE*SIZE));
//        System.out.println("#    T     mu     <E>     <sigma>   Cv    Susc");
        
        MU_lo-=Math.abs(in.m_MU_INC)/2.0; MU_hi+=Math.abs(in.m_MU_INC)/2.0;
        T_lo-=Math.abs(in.m_T_INC)/2.0; T_hi+=Math.abs(in.m_T_INC)/2.0;
        if(tempIS_inside_loop){
            //temp is the outer loop
            while((MU_lo<=mu)&&(mu<=MU_hi)){
                ArrayList innerL = new ArrayList();
                System.out.print("mu="+(new PrintfFormat("%8.3f").sprintf(mu))+" loop over T: ");
                while((T_lo<=temp)&&(temp<=T_hi)){
                    MCSampler mc = new MCSampler(config,temp,mu,in);
                    MCData d = mc.runMC();                    
//                System.out.println(d.m_T+" "+d.m_MU+" "+d.m_avgE+" "+d.m_avgS+" "+
//                         (d.m_avgE2-SIZE*SIZE*d.m_avgE*d.m_avgE)/(temp*temp)+" "+
//                         (d.m_avgS2-SIZE*SIZE*d.m_avgS*d.m_avgS)/(temp*temp));
                    config=d.m_config;
                    if(in.m_print_config){
                        DataPrinter cp = new DataPrinter(config,temp,mu,in);
                    }
                    if(in.m_print_allD){
                        DataPrinter dd = new DataPrinter(d,temp,mu,in);
                    }
                    d.clearArrays();
                    innerL.add(d);
                    System.out.print(new PrintfFormat("%8.3f").sprintf(temp));
                    temp+=in.m_T_INC;
                }
                System.out.print('\n');
                MCData[] alld = new MCData[innerL.size()];
                innerL.toArray(alld);
                DataPrinter dp = new DataPrinter(alld,mu,in,true);
                temp=in.m_T_START;
                mu+=in.m_MU_INC;
            }
            
        }else{
            //temp is the outer loop
            while((T_lo<=temp)&&(temp<=T_hi)){
                ArrayList innerL = new ArrayList();
                System.out.print("T="+(new PrintfFormat("%8.3f").sprintf(temp))+" loop over mu:");
                while((MU_lo<=mu)&&(mu<=MU_hi)){
                    MCSampler mc = new MCSampler(config,temp,mu,in);
                    MCData d = mc.runMC();
//                 System.out.println(d.m_T+" "+d.m_MU+" "+d.m_avgE+" "+d.m_avgS+" "+
//                         (d.m_avgE2-(SIZE*SIZE)*d.m_avgE*d.m_avgE)/(temp*temp)+" "+
//                         (d.m_avgS2-(SIZE*SIZE)*d.m_avgS*d.m_avgS)/(temp*temp));
                    System.out.print(new PrintfFormat("%8.3f").sprintf(mu));
                    config=d.m_config;
                    if(in.m_print_config){
                        DataPrinter cp = new DataPrinter(config,temp,mu,in);
                    }
                    if(in.m_print_allD){
                        DataPrinter dd = new DataPrinter(d,temp,mu,in);
                    }
                    d.clearArrays();
                    innerL.add(d);
                    mu+=in.m_MU_INC;
                }
                System.out.print('\n');
                MCData[] alld = new MCData[innerL.size()];
                innerL.toArray(alld);
                DataPrinter dp = new DataPrinter(alld,temp,in,false);
                mu=in.m_MU_START;
                temp+=in.m_T_INC;   
            }
        }
        DataPrinter dp = new DataPrinter(config,in.m_T_FINAL,in.m_MU_FINAL,in);
    }
    
    /**
     * returns a 2D array of integers size-by-size (+/- 1) according to config_id 
     * config_id=1 sets all spins -1
     * config_id=2 sets all spins +1
     * config_id=3 sets spins to a 2x2 ordering
     * config_id=4 yields a random configuration
     * @param size
     * @param config_id
     * @return
     */
    private static int[][] getConfig(int size,int config_id){
        Random rand = new Random();
        int[][] config = new int[size][size];
        for(int i=0; i<size; i++){
            for(int j=0; j<size; j++){
                if(config_id==1){
                    config[i][j]=-1;
                }else if(config_id==2){
                    config[i][j]=1;
                }else if(config_id==3){
                    if(((i+j)%2)==1){config[i][j]=1;}
                    if(((i+j)%2)==0){config[i][j]=-1;}
                }else if(config_id==4){
                    if(rand.nextDouble()<0.5){
                        config[i][j]=-1;
                    }else{
                        config[i][j]=1;
                    }                    
                }
                //System.out.print(" "+randconfig[i][j]);
            }
            //System.out.print('\n');
        }
        return config;
    }

}
