Require Import String List FunctionalExtensionality Omega Program.

(* These are the two types available in the language *)
Inductive type := 
| Global (* global *)
| Local (i : nat) (* local_i *).

(* Pointers need to be annotated with information about whether they are local
or global and what their id is *)
Inductive ptr := 
| G (n : nat) (* Global pointer to loc n *)
| L (n cid: nat) (* Local pointer to loc n in context cid *).

(* Program variable *)
Definition var := string.

(* Object field name *)
Definition field := string.

(* Object representation (maps field names to pointers) *)
Definition object := field -> ptr.

Inductive cell := Cell (p : ptr) (o : object).

(* Heap (either global or local) *)
Definition heap := list cell.


(* An empty object *)
Definition object0 : object := fun _ => G 0.


(*This is the grammar for the language*)

Inductive exp :=
(* First the expressions that can appear when you write the program *)
| Var (x : var)
| New 
| NewL 
| Seq (e1 e2 : exp)
| Let (x : var) (t : type) (e e' : exp)
| ReadField (e : exp) (f : field)
| WriteField (e : exp) (f : field) (e' : exp)
| Context (e: exp) 
(* And some constructs that can be introduced by the small step semantics *)
| Ptr (p:ptr)
| Inside (e: exp).



(** * The operational semantics *)

(* Substituting a pointer for a variable *)
(* This is the only kind of substitution we need for this language,
 * which should make it easier to prove the key lemma about substitution! *)
Fixpoint subst (x : var) (p : ptr) (e : exp) : exp :=
  match e with
    | Var y => if string_dec x y then Ptr p else e
    | New  => e
    | NewL  => e
    | Seq e1 e2 => Seq (subst x p e1) (subst x p e2)
    | Let y t e e' => Let y t (subst x p e) (if string_dec x y then e' else subst x p e')
    | ReadField e f => ReadField (subst x p e) f
    | WriteField e f e' => WriteField (subst x p e) f (subst x p e')
    | Context e => Context (subst x p e) 

    | Ptr _ => e
    | Inside e => Inside (subst x p e) 
  end.


Inductive localRedEnv := 
(* The small step semantics will be defined in terms of this *)
| LE (h : heap) (cid : nat) (oid : nat) (e : exp).


Inductive inHeap: heap -> ptr -> field -> ptr -> Prop :=
(*Predicate to tell us whether a value is stored in a particular location in the heap *)
| FoundR : forall p o h f,
  inHeap ((Cell p o)::h) p f (o f)
| LookingR: forall c h p f v,
  inHeap h p f v ->
  inHeap (c::h) p f v
.

Inductive heapWrite: heap->ptr->field->ptr->heap->Prop :=
(* The predicate heapWrite h p f v h' tells us whether the heap h' 
is the result of writing v to p.f in heap h. *)
| FoundW : forall p o h f v,
  heapWrite ((Cell p o)::h) p f v ((Cell p (fun x => if (string_dec x f) then v else (o x)) )::h)
| LookingW : forall h c p f v h',
  heapWrite h p f v h' ->
  heapWrite (c::h) p f v (c::h')
.

Inductive cleanupHeap: heap->nat->heap->Prop := 
(* cleanupHeap h c h' tells us whether h' is the result of removing 
from the heap all the local pointers with cid=c  *)
| EmptyCleanup: forall cid,
  cleanupHeap [] cid []
| FoundCleanup : forall n cid o h h', 
    cleanupHeap h cid h' ->
   cleanupHeap ((Cell (L n cid) o)::h) cid h'
| LocalNotFoundCleanup: forall n cid cid' o h h',
   cleanupHeap h cid h' -> cid <> cid' ->
   cleanupHeap ((Cell (L n cid') o)::h) cid ((Cell (L n cid') o)::h')
| GlobalNotFoundCleanup:
forall n cid h' o h,
   cleanupHeap h cid h' ->
   cleanupHeap ((Cell (G n) o)::h) cid ((Cell (G n) o)::h')
.

(* Basic local reductions *)
Inductive local : localRedEnv -> localRedEnv -> Prop := 
| LocalNew : forall h cid oid,
 local (LE h cid oid New ) (LE ((Cell (G oid) object0)::h) cid (oid+1) (Ptr (G oid)))
| LocalNewL : forall h cid oid,
 local (LE h cid oid NewL ) (LE ((Cell (L oid cid) object0)::h) cid (oid+1) (Ptr (L oid cid)))
| LocalLet : 
forall h cid oid x t p e,
 local (LE h cid oid (Let x  t (Ptr p) e) ) (LE h cid oid (subst x p e))
| LocalSeq : 
forall h cid oid p e, 
 local (LE h cid oid (Seq (Ptr p) e) ) (LE h cid oid e)
| LocalFRead:
forall h cid oid p f v, 
 (inHeap h p f v) ->
 local (LE h cid oid (ReadField (Ptr p) f) ) (LE h cid oid (Ptr v))
| LocalFWrite: 
forall h p f cid oid v h',
 (heapWrite h p f v h') ->
 local (LE h cid oid (WriteField (Ptr p) f (Ptr v)) ) (LE h' cid oid (Ptr v))
| LocalEnterContext:
forall h cid oid e,
 local (LE h cid oid (Context e))  (LE h (cid+1) oid (Inside e))
| LocalExitContext:
forall h h' cid oid v,
  (cleanupHeap h (cid+1) h') ->
  local (LE h (cid+1) oid (Inside (Ptr v)))  (LE h' cid oid (Ptr v))
.

Inductive evcontext := 
| EvHole
| EvSeq (h : evcontext) (e : exp)
| EvLet (x : var) (t:type) (h : evcontext) (e : exp)
| EvReadField (h : evcontext) (f : field)
| EvWriteField1 (h : evcontext) (f : field) (e : exp)
| EvWriteField2 (p : ptr) (f: field) (h : evcontext)
| EvInside (h :evcontext) 
.

Fixpoint plugEv (H : evcontext) (e0 : exp) : exp := 
match H with 
| EvHole => e0
| EvSeq h e => Seq (plugEv h e0) e
| EvLet x t h e => Let x t (plugEv h e0) e
| EvReadField h f => ReadField (plugEv h e0) f
| EvWriteField1 h f e => WriteField (plugEv h e0) f e
| EvWriteField2 p f h => WriteField (Ptr p) f (plugEv h e0)
| EvInside h => Inside (plugEv h e0)
end.

Inductive smallStep: localRedEnv->localRedEnv->Prop :=
| smallStepDef: 
forall H e h cid oid h' cid' oid' e' ein eout,
 local (LE h cid oid e) (LE h' cid' oid' e')->
ein = (plugEv H e) -> eout = (plugEv H e') 
-> smallStep (LE h cid oid ein) (LE h' cid' oid' eout)
.

