% Random graph construction routine with various models
% Gergana Bounova, October 31, 2005
function [adj] = random_graph(N,p,E,distribution,fun,degrees)
% INPUTS: 
% N - number of nodes
% p - probability, 0<=p<=1
% E - fixed number of edges
% distribution - probability distribution: use the
%                "connecting-stubs model"
%                generation model
% fun - customized pdf function, used only if distribution =
%       'custom'
% degrees - particular degree sequence, used only if distribution =
%           'sequence'
% OUTPUTS: adj - adjacency matrix of generated graph (symmetric)
adj=zeros(N);
% default is Erdos-Renyi graph:
switch nargin
 case 1
  p = 0.5; % default
  for i=1:N
    for j=i+1:N
      if rand<=p
	adj(i,j)=1; adj(j,i)=1;
      end
    end
  end
  
 case 2
  for i=1:N
    for j=i+1:N
      if rand<=p
	adj(i,j)=1; adj(j,i)=1;
      end
    end
  end
  
 case 3 % fix number of edges
  edges = 0;
  while edges < E
    i=ceil(rand*N); j=ceil(rand*N);
    if adj(i,j)==0 & not(i==j) % do not allow self-loops
      adj(i,j)=1; adj(j,i)=1;
      edges = edges + 1;
    end
  end
  
 otherwise % pick from a distribution
  % generate N random numbers from a distribution
  switch distribution
   case 'uniform'
    Nseq = 1;
    while mod(sum(Nseq),2)==1 % make sure # stubs is even
      Nseq = ceil((N-1)*rand(1,N));
    end
   case 'normal'
    Nseq = 1;
    while mod(sum(Nseq),2)==1 % make sure # stubs is even
      Nseq = ceil((N-1)/10*randn(1,N)+(N-1)/2);
    end
   case 'binomial'
    Nseq = 1;
    while mod(sum(Nseq),2)==1 % make sure # stubs is even
      Nseq = ceil(binornd(N-1,p,1,N));
    end
   case 'exponential'
    Nseq = 1;
    while mod(sum(Nseq),2)==1 % make sure # stubs is even
      Nseq = ceil(exprnd(N-1,1,N));
    end
   case 'geometric'
    Nseq = 1;
    while mod(sum(Nseq),2)==1 % make sure # stubs is even
      Nseq = ceil(geornd(p,1,N));
    end
   case 'custom'
    % pick a number from a custom pdf function
    % generate a random number x between 1 and N-1
    % accept it with probability fun(x)
    Nseq = 1;
    while mod(sum(Nseq),2)==1 % make sure # stubs is even
      Nseq = [];
      while length(Nseq)<N
	x = ceil(rand*(N-1));
	if rand <= fun(x)
	  Nseq = [Nseq x];
	end
      end
    end
   case 'sequence'
    Nseq = degrees;
  end  
  
  % connect stubs at random
  %edges = 0;
  nodes_left = [1:N];
  for i=1:N
    node(i).stubs = [1:Nseq(i)];
  end
  
  while numel(nodes_left)>0 % edges < sum(Nseq)/2
    
    randi = ceil(rand*length(nodes_left));
    nodei = nodes_left(randi);
    randj = ceil(rand*length(node(nodei).stubs));
    stubj = node(nodei).stubs(randj);
    
    randii = ceil(rand*length(nodes_left));
    nodeii = nodes_left(randii);
    randjj = ceil(rand*length(node(nodeii).stubs));
    stubjj = node(nodeii).stubs(randjj);
    
    % connect two nodes, as longs as stubs different
    if not(nodei==nodeii & stubj==stubjj)
      % add new links
      adj(nodei,nodeii) = adj(nodei,nodeii)+1; 
      adj(nodeii,nodei) = adj(nodei,nodeii);
      % remove connected stubs
      node(nodei).stubs = setdiff(node(nodei).stubs,stubj);
      node(nodeii).stubs = setdiff(node(nodeii).stubs,stubjj);
    end
    
    % purge empty nodes
    nodes_left1 = nodes_left;
    for i=1:length(nodes_left)
      if length(node(nodes_left(i)).stubs)==0
	nodes_left1 = setdiff(nodes_left1,nodes_left(i));
      end
    end
    nodes_left = nodes_left1;
  
  end
      
  
end  % end nargin options
